﻿using Hims.Api.Models;
using Hims.Api.Utilities;
using Hims.Domain.Repositories.UnitOfWork;
using Hims.Domain.Services;
using Hims.Shared.DataFilters;
using Hims.Shared.EntityModels;
using Hims.Shared.Library.Enums;
using Hims.Shared.UserModels;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Npgsql;
using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;
using System.Threading.Tasks;

namespace Hims.Api.Controllers
{
    /// <inheritdoc />
    [Authorize]
    [Route("api/session")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class SessionController : BaseController
    {
        /// <summary>
        /// The unit of work.
        /// </summary>
        private readonly IUnitOfWork unitOfWork;

        /// <summary>
        /// The coupon services.
        /// </summary>
        private readonly ISessionService sessionService;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public SessionController(ISessionService sessionService, IAuditLogService auditLogServices, IUnitOfWork unitOfWork)
        {
            this.sessionService = sessionService;
            this.auditLogServices = auditLogServices;
            this.unitOfWork = unitOfWork;
        }

        /// <summary>
        /// Fetches the asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<SessionModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody] SessionModel model)
        {
            model = (SessionModel)EmptyFilter.Handler(model);
            //model.LocationId = Convert.ToInt32(header.LocationId);
            var result = await this.sessionService.FetchAsync(model);
            foreach (var resource in result)
            {
                resource.StartTimeString = resource.StartTime.ToString();
                resource.EndTimeString = resource.EndTime.ToString();
                //try
                //{
                    
                //    resource.EndTimeString = resource.EndTime.ToString("hh:mm:ss", CultureInfo.CreateSpecificCulture("en-US"));
                //}
                //catch (Exception e)
                //{
                //    e.Message.ToString();
                //}
            }
            return result == null ? this.ServerError() : this.Success(result);
        }

        /// <summary>
        /// Adds the asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="header">The header.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("add")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> AddAsync([FromBody] SessionAddModel model)
        {
            model = (SessionAddModel)EmptyFilter.Handler(model);
            var response = await this.sessionService.AddAsync(model);
             if (response > 0)
             {
                var provider = await this.unitOfWork.Providers.FindAsync(m => m.ProviderId == model.ProviderId);
                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.Session,
                    LogFrom = (int)AccountType.Administrator,
                    LogDate = DateTime.Now,
                    LogDescription = $" { model.CreatedByName } has Created Sessions for {provider.FullName}  from {model.FromDate} to {model.ToDate}on { DateTime.Now }",
                    LocationId = model.LocationId
                };
                await this.auditLogServices.LogAsync(auditLogModel);

             }
            return this.Success(response);
            
        }

        /// <summary>
        /// Deletes the asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="header">The header.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("delete")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> DeleteAsync([FromBody] SessionModel model)
        {
            try
            {
                model = (SessionModel)EmptyFilter.Handler(model);
                var response = await this.sessionService.DeleteAsync(model);
                if (response == 0)
                {
                    return this.ServerError();
                }
                else if (response > 0)
                {
                    var provider = await this.unitOfWork.Providers.FindAsync(m => m.ProviderId == model.ProviderId);
                    var auditLogModel = new AuditLogModel
                    {
                        AccountId = model.ModifiedBy,
                        LogTypeId = (int)LogTypes.Session,
                        LogFrom = (int)AccountType.Administrator,
                        LogDate = DateTime.Now,
                        LogDescription = $" { model.ModifiedByName } has Deleted Session  for {provider.FullName} Date: {model.Date} on { DateTime.Now }",
                        LocationId = model.LocationId
                    };
                    await this.auditLogServices.LogAsync(auditLogModel);
                    
                }
                return this.Success("Session has been deleted successfully.");
            }
            catch (NpgsqlException exception)
            {
                if (exception.Message.Contains("violates foreign key constraint"))
                {
                    return this.Conflict("SessionType can't be deleted. Please contact Administrator.");
                }

                return this.ServerError();
            }
        }

        /// <summary>
        /// Modifies the status asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="header">The header.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("modify-status")]
        public async Task<ActionResult> ModifyStatusAsync([FromBody] SessionModel model)
        {
            model = (SessionModel)EmptyFilter.Handler(model);
            var response = await this.sessionService.ModifyStatusAsync(model);
            if (response>0)
            {
                var provider = await this.unitOfWork.Providers.FindAsync(m => m.ProviderId == model.ProviderId);
                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.ModifiedBy,
                    LogTypeId = (int)LogTypes.Session,
                    LogFrom = (int)AccountType.Administrator,
                    LogDate = DateTime.Now,
                    LogDescription = $" { model.ModifiedByName } has {((bool)model.Active ? "Activated" : "Deactivated")} the SessionType {provider.FullName} Date: {model.Date} on { DateTime.Now }",
                    LocationId = Convert.ToInt32(model.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLogModel);
            }
            
            return this.Success(response);
        }

        /// <summary>
        /// Updates the asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="header">The header.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("update")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody] SessionUpdateModel model)
        {
            model = (SessionUpdateModel)EmptyFilter.Handler(model);
            var response = await this.sessionService.UpdateAsync(model);
            if (response > 0)
            {
                var provider = await this.unitOfWork.Providers.FindAsync(m => m.ProviderId == model.ProviderId);
                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.ModifiedBy,
                    LogTypeId = (int)LogTypes.Session,
                    LogFrom = (int)AccountType.Administrator,
                    LogDate = DateTime.Now,
                    LogDescription = $" { model.ModifiedByName } has updated  Session for {provider.FullName} of {model.Date}  - {model.SessionId} on { DateTime.Now }",
                    LocationId = Convert.ToInt32(model.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLogModel);
            }
            return this.Success(response);

        }
    }
}
